const $schema = "https://raw.githubusercontent.com/martinring/tmlanguage/master/tmlanguage.json";
const name = "apl";
const scopeName = "source.apl";
const fileTypes = [
	"apl",
	"apla",
	"aplc",
	"aplf",
	"apli",
	"apln",
	"aplo",
	"dyalog",
	"dyapp",
	"mipage"
];
const firstLineMatch = "(?x)\n# APL symbols\n[⌶-⍺]\n|\n\n# Hashbang\n^\\#!.*(?:\\s|\\/|(?<=!)\\b)\n\t(?:gnu[-._]?apl|aplx?|dyalog)\n(?:$|\\s)\n|\n\n# Modeline\n(?i:\n\t# Emacs\n\t-\\*-(?:\\s*(?=[^:;\\s]+\\s*-\\*-)|(?:.*?[;\\s]|(?<=-\\*-))mode\\s*:\\s*)\n\t\tapl\n\t(?=[\\s;]|(?<![-*])-\\*-).*?-\\*-\n\n\t|\n\n\t# Vim\n\t(?:(?:\\s|^)vi(?:m[<=>]?\\d+|m)?|\\sex)(?=:(?=\\s*set?\\s[^\\n:]+:)|:(?!\\s* set?\\s))(?:(?:\\s|\\s*:\\s*)\\w*(?:\\s*=(?:[^\\n\\\\\\s]|\\\\.)*)?)*[\\s:](?:filetype|ft|syntax)\\s*=\n\t\tapl\n\t(?=\\s|:|$)\n)";
const foldingStartMarker = "{";
const foldingStopMarker = "}";
const patterns = [
	{
		match: "\\A#!.*$",
		name: "comment.line.shebang.apl"
	},
	{
		include: "#heredocs"
	},
	{
		include: "#main"
	},
	{
		contentName: "text.embedded.apl",
		begin: "^\\s*((\\))OFF|(\\])NEXTFILE)\\b(.*)$",
		end: "(?=N)A",
		beginCaptures: {
			"1": {
				name: "entity.name.command.eof.apl"
			},
			"2": {
				name: "punctuation.definition.command.apl"
			},
			"3": {
				name: "punctuation.definition.command.apl"
			},
			"4": {
				patterns: [
					{
						include: "#comment"
					}
				]
			}
		}
	},
	{
		name: "meta.round.bracketed.group.apl",
		patterns: [
			{
				include: "#main"
			}
		],
		begin: "\\(",
		end: "\\)",
		beginCaptures: {
			"0": {
				name: "punctuation.round.bracket.begin.apl"
			}
		},
		endCaptures: {
			"0": {
				name: "punctuation.round.bracket.end.apl"
			}
		}
	},
	{
		name: "meta.square.bracketed.group.apl",
		patterns: [
			{
				include: "#main"
			}
		],
		begin: "\\[",
		end: "\\]",
		beginCaptures: {
			"0": {
				name: "punctuation.square.bracket.begin.apl"
			}
		},
		endCaptures: {
			"0": {
				name: "punctuation.square.bracket.end.apl"
			}
		}
	},
	{
		name: "meta.system.command.apl",
		begin: "^\\s*((\\))\\S+)",
		end: "$",
		beginCaptures: {
			"1": {
				name: "entity.name.command.apl"
			},
			"2": {
				name: "punctuation.definition.command.apl"
			}
		},
		patterns: [
			{
				include: "#command-arguments"
			},
			{
				include: "#command-switches"
			},
			{
				include: "#main"
			}
		]
	},
	{
		name: "meta.user.command.apl",
		begin: "^\\s*((\\])\\S+)",
		end: "$",
		beginCaptures: {
			"1": {
				name: "entity.name.command.apl"
			},
			"2": {
				name: "punctuation.definition.command.apl"
			}
		},
		patterns: [
			{
				include: "#command-arguments"
			},
			{
				include: "#command-switches"
			},
			{
				include: "#main"
			}
		]
	}
];
const repository = {
	main: {
		patterns: [
			{
				include: "#class"
			},
			{
				include: "#definition"
			},
			{
				include: "#comment"
			},
			{
				include: "#label"
			},
			{
				include: "#sck"
			},
			{
				include: "#strings"
			},
			{
				include: "#number"
			},
			{
				include: "#lambda"
			},
			{
				include: "#sysvars"
			},
			{
				include: "#symbols"
			},
			{
				include: "#name"
			}
		]
	},
	comment: {
		patterns: [
			{
				name: "comment.line.apl",
				begin: "⍝",
				end: "$",
				captures: {
					"0": {
						name: "punctuation.definition.comment.apl"
					}
				}
			}
		]
	},
	number: {
		patterns: [
			{
				name: "constant.numeric.apl",
				match: "¯?[0-9][¯0-9A-Za-z]*(?:\\.[¯0-9Ee][¯0-9A-Za-z]*)*|¯?\\.[0-9Ee][¯0-9A-Za-z]*"
			}
		]
	},
	strings: {
		patterns: [
			{
				name: "string.quoted.single.apl",
				begin: "'",
				end: "'|$",
				beginCaptures: {
					"0": {
						name: "punctuation.definition.string.begin.apl"
					}
				},
				endCaptures: {
					"0": {
						name: "punctuation.definition.string.end.apl"
					}
				},
				patterns: [
					{
						match: "[^']*[^'\\n\\r\\\\]$",
						name: "invalid.illegal.string.apl"
					}
				]
			},
			{
				name: "string.quoted.double.apl",
				begin: "\"",
				end: "\"|$",
				beginCaptures: {
					"0": {
						name: "punctuation.definition.string.begin.apl"
					}
				},
				endCaptures: {
					"0": {
						name: "punctuation.definition.string.end.apl"
					}
				},
				patterns: [
					{
						match: "[^\"]*[^\"\\n\\r\\\\]$",
						name: "invalid.illegal.string.apl"
					}
				]
			}
		]
	},
	name: {
		patterns: [
			{
				name: "variable.other.readwrite.apl",
				match: "(?x)\n[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ]\n[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]*"
			}
		]
	},
	label: {
		patterns: [
			{
				name: "meta.label.apl",
				match: "(?x)\n^\\s*\n(\n\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ]\n\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]*\n)\n(:)",
				captures: {
					"1": {
						name: "entity.label.name.apl"
					},
					"2": {
						name: "punctuation.definition.label.end.apl"
					}
				}
			}
		]
	},
	symbols: {
		patterns: [
			{
				match: "(?<=\\s)←(?=\\s|$)",
				name: "keyword.spaced.operator.assignment.apl"
			},
			{
				match: "(?<=\\s)→(?=\\s|$)",
				name: "keyword.spaced.control.goto.apl"
			},
			{
				match: "(?<=\\s)≡(?=\\s|$)",
				name: "keyword.spaced.operator.identical.apl"
			},
			{
				match: "(?<=\\s)≢(?=\\s|$)",
				name: "keyword.spaced.operator.not-identical.apl"
			},
			{
				match: "\\+",
				name: "keyword.operator.plus.apl"
			},
			{
				match: "[-−]",
				name: "keyword.operator.minus.apl"
			},
			{
				match: "×",
				name: "keyword.operator.times.apl"
			},
			{
				match: "÷",
				name: "keyword.operator.divide.apl"
			},
			{
				match: "⌊",
				name: "keyword.operator.floor.apl"
			},
			{
				match: "⌈",
				name: "keyword.operator.ceiling.apl"
			},
			{
				match: "[∣|]",
				name: "keyword.operator.absolute.apl"
			},
			{
				match: "[⋆*]",
				name: "keyword.operator.exponent.apl"
			},
			{
				match: "⍟",
				name: "keyword.operator.logarithm.apl"
			},
			{
				match: "○",
				name: "keyword.operator.circle.apl"
			},
			{
				match: "!",
				name: "keyword.operator.factorial.apl"
			},
			{
				match: "∧",
				name: "keyword.operator.and.apl"
			},
			{
				match: "∨",
				name: "keyword.operator.or.apl"
			},
			{
				match: "⍲",
				name: "keyword.operator.nand.apl"
			},
			{
				match: "⍱",
				name: "keyword.operator.nor.apl"
			},
			{
				match: "<",
				name: "keyword.operator.less.apl"
			},
			{
				match: "≤",
				name: "keyword.operator.less-or-equal.apl"
			},
			{
				match: "=",
				name: "keyword.operator.equal.apl"
			},
			{
				match: "≥",
				name: "keyword.operator.greater-or-equal.apl"
			},
			{
				match: ">",
				name: "keyword.operator.greater.apl"
			},
			{
				match: "≠",
				name: "keyword.operator.not-equal.apl"
			},
			{
				match: "[∼~]",
				name: "keyword.operator.tilde.apl"
			},
			{
				match: "\\?",
				name: "keyword.operator.random.apl"
			},
			{
				match: "[∊∈]",
				name: "keyword.operator.member-of.apl"
			},
			{
				match: "⍷",
				name: "keyword.operator.find.apl"
			},
			{
				match: ",",
				name: "keyword.operator.comma.apl"
			},
			{
				match: "⍪",
				name: "keyword.operator.comma-bar.apl"
			},
			{
				match: "⌷",
				name: "keyword.operator.squad.apl"
			},
			{
				match: "⍳",
				name: "keyword.operator.iota.apl"
			},
			{
				match: "⍴",
				name: "keyword.operator.rho.apl"
			},
			{
				match: "↑",
				name: "keyword.operator.take.apl"
			},
			{
				match: "↓",
				name: "keyword.operator.drop.apl"
			},
			{
				match: "⊣",
				name: "keyword.operator.left.apl"
			},
			{
				match: "⊢",
				name: "keyword.operator.right.apl"
			},
			{
				match: "⊤",
				name: "keyword.operator.encode.apl"
			},
			{
				match: "⊥",
				name: "keyword.operator.decode.apl"
			},
			{
				match: "\\/",
				name: "keyword.operator.slash.apl"
			},
			{
				match: "⌿",
				name: "keyword.operator.slash-bar.apl"
			},
			{
				match: "\\x5C",
				name: "keyword.operator.backslash.apl"
			},
			{
				match: "⍀",
				name: "keyword.operator.backslash-bar.apl"
			},
			{
				match: "⌽",
				name: "keyword.operator.rotate-last.apl"
			},
			{
				match: "⊖",
				name: "keyword.operator.rotate-first.apl"
			},
			{
				match: "⍉",
				name: "keyword.operator.transpose.apl"
			},
			{
				match: "⍋",
				name: "keyword.operator.grade-up.apl"
			},
			{
				match: "⍒",
				name: "keyword.operator.grade-down.apl"
			},
			{
				match: "⌹",
				name: "keyword.operator.quad-divide.apl"
			},
			{
				match: "≡",
				name: "keyword.operator.identical.apl"
			},
			{
				match: "≢",
				name: "keyword.operator.not-identical.apl"
			},
			{
				match: "⊂",
				name: "keyword.operator.enclose.apl"
			},
			{
				match: "⊃",
				name: "keyword.operator.pick.apl"
			},
			{
				match: "∩",
				name: "keyword.operator.intersection.apl"
			},
			{
				match: "∪",
				name: "keyword.operator.union.apl"
			},
			{
				match: "⍎",
				name: "keyword.operator.hydrant.apl"
			},
			{
				match: "⍕",
				name: "keyword.operator.thorn.apl"
			},
			{
				match: "⊆",
				name: "keyword.operator.underbar-shoe-left.apl"
			},
			{
				match: "⍸",
				name: "keyword.operator.underbar-iota.apl"
			},
			{
				match: "¨",
				name: "keyword.operator.each.apl"
			},
			{
				match: "⍤",
				name: "keyword.operator.rank.apl"
			},
			{
				match: "⌸",
				name: "keyword.operator.quad-equal.apl"
			},
			{
				match: "⍨",
				name: "keyword.operator.commute.apl"
			},
			{
				match: "⍣",
				name: "keyword.operator.power.apl"
			},
			{
				match: "\\.",
				name: "keyword.operator.dot.apl"
			},
			{
				match: "∘",
				name: "keyword.operator.jot.apl"
			},
			{
				match: "⍠",
				name: "keyword.operator.quad-colon.apl"
			},
			{
				match: "&",
				name: "keyword.operator.ampersand.apl"
			},
			{
				match: "⌶",
				name: "keyword.operator.i-beam.apl"
			},
			{
				match: "⌺",
				name: "keyword.operator.quad-diamond.apl"
			},
			{
				match: "@",
				name: "keyword.operator.at.apl"
			},
			{
				match: "◊",
				name: "keyword.operator.lozenge.apl"
			},
			{
				match: ";",
				name: "keyword.operator.semicolon.apl"
			},
			{
				match: "¯",
				name: "keyword.operator.high-minus.apl"
			},
			{
				match: "←",
				name: "keyword.operator.assignment.apl"
			},
			{
				match: "→",
				name: "keyword.control.goto.apl"
			},
			{
				match: "⍬",
				name: "constant.language.zilde.apl"
			},
			{
				match: "⋄",
				name: "keyword.operator.diamond.apl"
			},
			{
				match: "⍫",
				name: "keyword.operator.lock.apl"
			},
			{
				match: "⎕",
				name: "keyword.operator.quad.apl"
			},
			{
				match: "##",
				name: "constant.language.namespace.parent.apl"
			},
			{
				match: "#",
				name: "constant.language.namespace.root.apl"
			},
			{
				match: "⌻",
				name: "keyword.operator.quad-jot.apl"
			},
			{
				match: "⌼",
				name: "keyword.operator.quad-circle.apl"
			},
			{
				match: "⌾",
				name: "keyword.operator.circle-jot.apl"
			},
			{
				match: "⍁",
				name: "keyword.operator.quad-slash.apl"
			},
			{
				match: "⍂",
				name: "keyword.operator.quad-backslash.apl"
			},
			{
				match: "⍃",
				name: "keyword.operator.quad-less.apl"
			},
			{
				match: "⍄",
				name: "keyword.operator.greater.apl"
			},
			{
				match: "⍅",
				name: "keyword.operator.vane-left.apl"
			},
			{
				match: "⍆",
				name: "keyword.operator.vane-right.apl"
			},
			{
				match: "⍇",
				name: "keyword.operator.quad-arrow-left.apl"
			},
			{
				match: "⍈",
				name: "keyword.operator.quad-arrow-right.apl"
			},
			{
				match: "⍊",
				name: "keyword.operator.tack-down.apl"
			},
			{
				match: "⍌",
				name: "keyword.operator.quad-caret-down.apl"
			},
			{
				match: "⍍",
				name: "keyword.operator.quad-del-up.apl"
			},
			{
				match: "⍏",
				name: "keyword.operator.vane-up.apl"
			},
			{
				match: "⍐",
				name: "keyword.operator.quad-arrow-up.apl"
			},
			{
				match: "⍑",
				name: "keyword.operator.tack-up.apl"
			},
			{
				match: "⍓",
				name: "keyword.operator.quad-caret-up.apl"
			},
			{
				match: "⍔",
				name: "keyword.operator.quad-del-down.apl"
			},
			{
				match: "⍖",
				name: "keyword.operator.vane-down.apl"
			},
			{
				match: "⍗",
				name: "keyword.operator.quad-arrow-down.apl"
			},
			{
				match: "⍘",
				name: "keyword.operator.underbar-quote.apl"
			},
			{
				match: "⍚",
				name: "keyword.operator.underbar-diamond.apl"
			},
			{
				match: "⍛",
				name: "keyword.operator.underbar-jot.apl"
			},
			{
				match: "⍜",
				name: "keyword.operator.underbar-circle.apl"
			},
			{
				match: "⍞",
				name: "keyword.operator.quad-quote.apl"
			},
			{
				match: "⍡",
				name: "keyword.operator.dotted-tack-up.apl"
			},
			{
				match: "⍢",
				name: "keyword.operator.dotted-del.apl"
			},
			{
				match: "⍥",
				name: "keyword.operator.dotted-circle.apl"
			},
			{
				match: "⍦",
				name: "keyword.operator.stile-shoe-up.apl"
			},
			{
				match: "⍧",
				name: "keyword.operator.stile-shoe-left.apl"
			},
			{
				match: "⍩",
				name: "keyword.operator.dotted-greater.apl"
			},
			{
				match: "⍭",
				name: "keyword.operator.stile-tilde.apl"
			},
			{
				match: "⍮",
				name: "keyword.operator.underbar-semicolon.apl"
			},
			{
				match: "⍯",
				name: "keyword.operator.quad-not-equal.apl"
			},
			{
				match: "⍰",
				name: "keyword.operator.quad-question.apl"
			}
		]
	},
	definition: {
		patterns: [
			{
				name: "meta.function.apl",
				begin: "(?x) ^\\s*? (?# 1: keyword.operator.nabla.apl) (∇) (?: \\s* (?: (?# 2: entity.function.return-value.apl) ( [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* ) | \\s* (?# 3: entity.function.return-value.shy.apl) ( (\\{)             (?# 4: punctuation.definition.return-value.begin.apl) (?: \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* )* (\\})             (?# 5: punctuation.definition.return-value.end.apl) | (\\()             (?# 6: punctuation.definition.return-value.begin.apl) (?: \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* )* (\\))             (?# 7: punctuation.definition.return-value.end.apl) | (\\(\\s*\\{)      (?# 8: punctuation.definition.return-value.begin.apl) (?: \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* )* (\\}\\s*\\))      (?# 9: punctuation.definition.return-value.end.apl) | (\\{\\s*\\()      (?# 10: punctuation.definition.return-value.begin.apl) (?: \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* )* (\\)\\s*\\})      (?# 11: punctuation.definition.return-value.end.apl) ) \\s* ) \\s* (?# 12: keyword.operator.assignment.apl) (←) )? \\s* (?: (?# MONADIC) (?: (?# 13: entity.function.name.apl) ( [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* ) \\s* (?# 14: entity.function.axis.apl) ( (?# 15: punctuation.definition.axis.begin.apl) (\\[) \\s* (?: \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* (?# 16: invalid.illegal.extra-characters.apl) (.*?) | (?# 17: invalid.illegal.apl) ([^\\]]*) ) \\s* (?# 18: punctuation.definition.axis.end.apl) (\\]) )? \\s*? (?# 19: entity.function.arguments.right.apl) ( (?<=\\s|\\]) [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* | (\\()   (?# 20: punctuation.definition.arguments.begin.apl) (?: \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* )* (\\))   (?# 21: punctuation.definition.arguments.end.apl) ) \\s* (?=;|$) ) | (?# DYADIC/AMBIVALENT) (?#==================) (?: (?# 22: entity.function.arguments.left.apl) ( [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s+ ) | (?# 23: entity.function.arguments.left.optional.apl) ( (\\{)          (?# 24: punctuation.definition.arguments.begin.apl) (?: \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* )* (\\})          (?# 25: punctuation.definition.arguments.end.apl) | (\\(\\s*\\{)   (?# 26: punctuation.definition.arguments.begin.apl) (?: \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* )* (\\}\\s*\\))   (?# 27: punctuation.definition.arguments.end.apl) | (\\{\\s*\\()   (?# 28: punctuation.definition.arguments.begin.apl) (?: \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* )* (\\)\\s*\\})   (?# 29: punctuation.definition.arguments.end.apl) ) )? \\s* (?: (?# 30: entity.function.name.apl) ( [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* ) \\s* (?# 31: entity.function.axis.apl) ( (?# 32: punctuation.definition.axis.begin.apl) (\\[) \\s* (?: \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* (?# 33: invalid.illegal.extra-characters.apl) (.*?) | (?# 34: invalid.illegal.apl) ([^\\]]*) ) \\s* (?# 35: punctuation.definition.axis.end.apl) (\\]) )? | (?# 36: entity.function.operands.apl) ( (?# 37: punctuation.definition.operands.begin.apl) (\\() (?# 38: entity.function.operands.left.apl) ( \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* )? \\s* (?# 39: entity.function.name.apl) ( [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* ) \\s*? (?# 40: entity.function.axis.apl) ( (?# 41: punctuation.definition.axis.begin.apl) (\\[) \\s* (?: \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* (?# 42: invalid.illegal.extra-characters.apl) (.*?) | (?# 43: invalid.illegal.apl) ([^\\]]*) ) \\s* (?# 44: punctuation.definition.axis.end.apl) (\\]) )? \\s* (?# 45: entity.function.operands.right.apl) ( [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* )? (?# 46: punctuation.definition.operands.end.apl) (\\)) ) ) \\s* (?# 47: entity.function.arguments.right.apl) ( (?<=\\s|\\]) [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* | \\s* (\\()   (?# 48: punctuation.definition.arguments.begin.apl) (?: \\s* [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* )* (\\))   (?# 49: punctuation.definition.arguments.end.apl) )? (?#==================) ) \\s* (?# 50: invalid.illegal.arguments.right.apl) ([^;]+)? (?# 51: entity.function.local-variables.apl) ( (?# 52: Include “;”) ( (?> \\s* ; (?: \\s* [⎕A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ] [A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]* \\s* )+ )+ ) | (?# 53: invalid.illegal.local-variables.apl) ([^⍝]+) )? \\s* (?# 54: comment.line.apl) (⍝.*)? $",
				end: "^\\s*?(?:(∇)|(⍫))\\s*?(⍝.*?)?$",
				patterns: [
					{
						name: "entity.function.definition.apl",
						match: "(?x)\n^\\s*\n(\n\t(?>\n\t\t;\n\t\t(?:\n\t\t\t\\s*\n\t\t\t[⎕A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ]\n\t\t\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]*\n\t\t\t\\s*\n\t\t)+\n\t)+\n)",
						captures: {
							"0": {
								name: "entity.function.local-variables.apl"
							},
							"1": {
								patterns: [
									{
										name: "punctuation.separator.apl",
										match: ";"
									}
								]
							}
						}
					},
					{
						include: "$self"
					}
				],
				beginCaptures: {
					"0": {
						name: "entity.function.definition.apl"
					},
					"1": {
						name: "keyword.operator.nabla.apl"
					},
					"2": {
						name: "entity.function.return-value.apl"
					},
					"3": {
						name: "entity.function.return-value.shy.apl"
					},
					"4": {
						name: "punctuation.definition.return-value.begin.apl"
					},
					"5": {
						name: "punctuation.definition.return-value.end.apl"
					},
					"6": {
						name: "punctuation.definition.return-value.begin.apl"
					},
					"7": {
						name: "punctuation.definition.return-value.end.apl"
					},
					"8": {
						name: "punctuation.definition.return-value.begin.apl"
					},
					"9": {
						name: "punctuation.definition.return-value.end.apl"
					},
					"10": {
						name: "punctuation.definition.return-value.begin.apl"
					},
					"11": {
						name: "punctuation.definition.return-value.end.apl"
					},
					"12": {
						name: "keyword.operator.assignment.apl"
					},
					"13": {
						name: "entity.function.name.apl",
						patterns: [
							{
								include: "#embolden"
							}
						]
					},
					"14": {
						name: "entity.function.axis.apl"
					},
					"15": {
						name: "punctuation.definition.axis.begin.apl"
					},
					"16": {
						name: "invalid.illegal.extra-characters.apl"
					},
					"17": {
						name: "invalid.illegal.apl"
					},
					"18": {
						name: "punctuation.definition.axis.end.apl"
					},
					"19": {
						name: "entity.function.arguments.right.apl"
					},
					"20": {
						name: "punctuation.definition.arguments.begin.apl"
					},
					"21": {
						name: "punctuation.definition.arguments.end.apl"
					},
					"22": {
						name: "entity.function.arguments.left.apl"
					},
					"23": {
						name: "entity.function.arguments.left.optional.apl"
					},
					"24": {
						name: "punctuation.definition.arguments.begin.apl"
					},
					"25": {
						name: "punctuation.definition.arguments.end.apl"
					},
					"26": {
						name: "punctuation.definition.arguments.begin.apl"
					},
					"27": {
						name: "punctuation.definition.arguments.end.apl"
					},
					"28": {
						name: "punctuation.definition.arguments.begin.apl"
					},
					"29": {
						name: "punctuation.definition.arguments.end.apl"
					},
					"30": {
						name: "entity.function.name.apl",
						patterns: [
							{
								include: "#embolden"
							}
						]
					},
					"31": {
						name: "entity.function.axis.apl"
					},
					"32": {
						name: "punctuation.definition.axis.begin.apl"
					},
					"33": {
						name: "invalid.illegal.extra-characters.apl"
					},
					"34": {
						name: "invalid.illegal.apl"
					},
					"35": {
						name: "punctuation.definition.axis.end.apl"
					},
					"36": {
						name: "entity.function.operands.apl"
					},
					"37": {
						name: "punctuation.definition.operands.begin.apl"
					},
					"38": {
						name: "entity.function.operands.left.apl"
					},
					"39": {
						name: "entity.function.name.apl",
						patterns: [
							{
								include: "#embolden"
							}
						]
					},
					"40": {
						name: "entity.function.axis.apl"
					},
					"41": {
						name: "punctuation.definition.axis.begin.apl"
					},
					"42": {
						name: "invalid.illegal.extra-characters.apl"
					},
					"43": {
						name: "invalid.illegal.apl"
					},
					"44": {
						name: "punctuation.definition.axis.end.apl"
					},
					"45": {
						name: "entity.function.operands.right.apl"
					},
					"46": {
						name: "punctuation.definition.operands.end.apl"
					},
					"47": {
						name: "entity.function.arguments.right.apl"
					},
					"48": {
						name: "punctuation.definition.arguments.begin.apl"
					},
					"49": {
						name: "punctuation.definition.arguments.end.apl"
					},
					"50": {
						name: "invalid.illegal.arguments.right.apl"
					},
					"51": {
						name: "entity.function.local-variables.apl"
					},
					"52": {
						patterns: [
							{
								name: "punctuation.separator.apl",
								match: ";"
							}
						]
					},
					"53": {
						name: "invalid.illegal.local-variables.apl"
					},
					"54": {
						name: "comment.line.apl"
					}
				},
				endCaptures: {
					"1": {
						name: "keyword.operator.nabla.apl"
					},
					"2": {
						name: "keyword.operator.lock.apl"
					},
					"3": {
						name: "comment.line.apl"
					}
				}
			}
		]
	},
	embolden: {
		patterns: [
			{
				name: "markup.bold.identifier.apl",
				match: ".+"
			}
		]
	},
	lambda: {
		name: "meta.lambda.function.apl",
		begin: "\\{",
		end: "\\}",
		beginCaptures: {
			"0": {
				name: "punctuation.definition.lambda.begin.apl"
			}
		},
		endCaptures: {
			"0": {
				name: "punctuation.definition.lambda.end.apl"
			}
		},
		patterns: [
			{
				include: "#main"
			},
			{
				include: "#lambda-variables"
			}
		]
	},
	"lambda-variables": {
		patterns: [
			{
				match: "⍺⍺",
				name: "constant.language.lambda.operands.left.apl"
			},
			{
				match: "⍵⍵",
				name: "constant.language.lambda.operands.right.apl"
			},
			{
				match: "[⍺⍶]",
				name: "constant.language.lambda.arguments.left.apl"
			},
			{
				match: "[⍵⍹]",
				name: "constant.language.lambda.arguments.right.apl"
			},
			{
				match: "χ",
				name: "constant.language.lambda.arguments.axis.apl"
			},
			{
				match: "∇∇",
				name: "constant.language.lambda.operands.self.operator.apl"
			},
			{
				match: "∇",
				name: "constant.language.lambda.operands.self.function.apl"
			},
			{
				match: "λ",
				name: "constant.language.lambda.symbol.apl"
			}
		]
	},
	sysvars: {
		patterns: [
			{
				match: "(?:(⎕)|(⍞))[A-Za-z]*",
				name: "support.system.variable.apl",
				captures: {
					"1": {
						name: "punctuation.definition.quad.apl"
					},
					"2": {
						name: "punctuation.definition.quad-quote.apl"
					}
				}
			}
		]
	},
	"command-arguments": {
		patterns: [
			{
				name: "variable.parameter.argument.apl",
				begin: "\\b(?=\\S)",
				end: "\\b(?=\\s)",
				patterns: [
					{
						include: "#main"
					}
				]
			}
		]
	},
	"command-switches": {
		patterns: [
			{
				name: "variable.parameter.switch.apl",
				begin: "(?x)\n(?<=\\s)(-)\n(\n\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ]\n\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]*\n)\n(=)",
				end: "\\b(?=\\s)",
				beginCaptures: {
					"1": {
						name: "punctuation.delimiter.switch.apl"
					},
					"2": {
						name: "entity.name.switch.apl"
					},
					"3": {
						name: "punctuation.assignment.switch.apl"
					}
				},
				patterns: [
					{
						include: "#main"
					}
				]
			},
			{
				name: "variable.parameter.switch.apl",
				match: "(?x)\n(?<=\\s)(-)\n(\n\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ]\n\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]*\n)\n(?!=)",
				captures: {
					"1": {
						name: "punctuation.delimiter.switch.apl"
					},
					"2": {
						name: "entity.name.switch.apl"
					}
				}
			}
		]
	},
	sck: {
		patterns: [
			{
				name: "keyword.control.sck.apl",
				match: "(?<=\\s|^)(:)[A-Za-z]+",
				captures: {
					"1": {
						name: "punctuation.definition.sck.begin.apl"
					}
				}
			}
		]
	},
	"class": {
		patterns: [
			{
				begin: "(?x)\n(?<=\\s|^)\n((:)Class)\n\\s+\n(\n\t'[^']*'?\n\t|\n\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ]\n\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]*\n)\n\\s*\n(\n\t(:)\n\t\\s*\n\t(?:\n\t\t(\n\t\t\t'[^']*'?\n\t\t\t|\n\t\t\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ]\n\t\t\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]*\n\t\t)\n\t\t\\s*\n\t)?\n)?\n(.*?)$",
				end: "(?<=\\s|^)((:)EndClass)(?=\\b)",
				beginCaptures: {
					"0": {
						name: "meta.class.apl"
					},
					"1": {
						name: "keyword.control.class.apl"
					},
					"2": {
						name: "punctuation.definition.class.apl"
					},
					"3": {
						name: "entity.name.type.class.apl",
						patterns: [
							{
								include: "#strings"
							}
						]
					},
					"4": {
						name: "entity.other.inherited-class.apl"
					},
					"5": {
						name: "punctuation.separator.inheritance.apl"
					},
					"6": {
						patterns: [
							{
								include: "#strings"
							}
						]
					},
					"7": {
						name: "entity.other.class.interfaces.apl",
						patterns: [
							{
								include: "#csv"
							}
						]
					}
				},
				endCaptures: {
					"1": {
						name: "keyword.control.class.apl"
					},
					"2": {
						name: "punctuation.definition.class.apl"
					}
				},
				patterns: [
					{
						name: "meta.field.apl",
						begin: "(?<=\\s|^)(:)Field(?=\\s)",
						end: "\\s*(←.*)?(?:$|(?=⍝))",
						beginCaptures: {
							"0": {
								name: "keyword.control.field.apl"
							},
							"1": {
								name: "punctuation.definition.field.apl"
							}
						},
						endCaptures: {
							"0": {
								name: "entity.other.initial-value.apl"
							},
							"1": {
								patterns: [
									{
										include: "#main"
									}
								]
							}
						},
						patterns: [
							{
								name: "storage.modifier.access.public.apl",
								match: "(?<=\\s|^)Public(?=\\s|$)"
							},
							{
								name: "storage.modifier.access.private.apl",
								match: "(?<=\\s|^)Private(?=\\s|$)"
							},
							{
								name: "storage.modifier.shared.apl",
								match: "(?<=\\s|^)Shared(?=\\s|$)"
							},
							{
								name: "storage.modifier.instance.apl",
								match: "(?<=\\s|^)Instance(?=\\s|$)"
							},
							{
								name: "storage.modifier.readonly.apl",
								match: "(?<=\\s|^)ReadOnly(?=\\s|$)"
							},
							{
								name: "entity.name.type.apl",
								match: "(?x)\n(\n\t'[^']*'?\n\t|\n\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ]\n\t[A-Z_a-zÀ-ÖØ-Ýßà-öø-üþ∆⍙Ⓐ-Ⓩ¯0-9]*\n)",
								captures: {
									"1": {
										patterns: [
											{
												include: "#strings"
											}
										]
									}
								}
							}
						]
					},
					{
						include: "$self"
					}
				]
			}
		]
	},
	csv: {
		patterns: [
			{
				match: ",",
				name: "punctuation.separator.apl"
			},
			{
				include: "$self"
			}
		]
	},
	heredocs: {
		patterns: [
			{
				name: "meta.heredoc.apl",
				begin: "^.*?⎕INP\\s+('|\")((?i).*?HTML?.*?|END-OF-⎕INP)\\1.*$",
				end: "^.*?\\2.*?$",
				beginCaptures: {
					"0": {
						patterns: [
							{
								include: "#main"
							}
						]
					}
				},
				endCaptures: {
					"0": {
						name: "constant.other.apl"
					}
				},
				contentName: "text.embedded.html.basic",
				patterns: [
					{
						include: "text.html.basic"
					},
					{
						include: "#embedded-apl"
					}
				]
			},
			{
				name: "meta.heredoc.apl",
				begin: "^.*?⎕INP\\s+('|\")((?i).*?(?:XML|XSLT|SVG|RSS).*?)\\1.*$",
				end: "^.*?\\2.*?$",
				beginCaptures: {
					"0": {
						patterns: [
							{
								include: "#main"
							}
						]
					}
				},
				endCaptures: {
					"0": {
						name: "constant.other.apl"
					}
				},
				contentName: "text.embedded.xml",
				patterns: [
					{
						include: "text.xml"
					},
					{
						include: "#embedded-apl"
					}
				]
			},
			{
				name: "meta.heredoc.apl",
				begin: "^.*?⎕INP\\s+('|\")((?i).*?(?:CSS|stylesheet).*?)\\1.*$",
				end: "^.*?\\2.*?$",
				beginCaptures: {
					"0": {
						patterns: [
							{
								include: "#main"
							}
						]
					}
				},
				endCaptures: {
					"0": {
						name: "constant.other.apl"
					}
				},
				contentName: "source.embedded.css",
				patterns: [
					{
						include: "source.css"
					},
					{
						include: "#embedded-apl"
					}
				]
			},
			{
				name: "meta.heredoc.apl",
				begin: "^.*?⎕INP\\s+('|\")((?i).*?(?:JS(?!ON)|(?:ECMA|J|Java).?Script).*?)\\1.*$",
				end: "^.*?\\2.*?$",
				beginCaptures: {
					"0": {
						patterns: [
							{
								include: "#main"
							}
						]
					}
				},
				endCaptures: {
					"0": {
						name: "constant.other.apl"
					}
				},
				contentName: "source.embedded.js",
				patterns: [
					{
						include: "source.js"
					},
					{
						include: "#embedded-apl"
					}
				]
			},
			{
				name: "meta.heredoc.apl",
				begin: "^.*?⎕INP\\s+('|\")((?i).*?(?:JSON).*?)\\1.*$",
				end: "^.*?\\2.*?$",
				beginCaptures: {
					"0": {
						patterns: [
							{
								include: "#main"
							}
						]
					}
				},
				endCaptures: {
					"0": {
						name: "constant.other.apl"
					}
				},
				contentName: "source.embedded.json",
				patterns: [
					{
						include: "source.json"
					},
					{
						include: "#embedded-apl"
					}
				]
			},
			{
				name: "meta.heredoc.apl",
				begin: "^.*?⎕INP\\s+('|\")(?i)((?:Raw|Plain)?\\s*Te?xt)\\1.*$",
				end: "^.*?\\2.*?$",
				beginCaptures: {
					"0": {
						patterns: [
							{
								include: "#main"
							}
						]
					}
				},
				endCaptures: {
					"0": {
						name: "constant.other.apl"
					}
				},
				contentName: "text.embedded.plain",
				patterns: [
					{
						include: "#embedded-apl"
					}
				]
			},
			{
				name: "meta.heredoc.apl",
				begin: "^.*?⎕INP\\s+('|\")(.*?)\\1.*$",
				end: "^.*?\\2.*?$",
				beginCaptures: {
					"0": {
						patterns: [
							{
								include: "#main"
							}
						]
					}
				},
				endCaptures: {
					"0": {
						name: "constant.other.apl"
					}
				},
				patterns: [
					{
						include: "$self"
					}
				]
			}
		]
	},
	"embedded-apl": {
		patterns: [
			{
				name: "meta.embedded.block.apl",
				begin: "(?i)(<(\\?|%)(?:apl(?=\\s+)|=))",
				end: "(?<=\\s)(\\2>)",
				patterns: [
					{
						include: "#main"
					}
				],
				beginCaptures: {
					"1": {
						name: "punctuation.section.embedded.begin.apl"
					}
				},
				endCaptures: {
					"1": {
						name: "punctuation.section.embedded.end.apl"
					}
				}
			}
		]
	}
};
const apl_tmLanguage = {
	$schema: $schema,
	name: name,
	scopeName: scopeName,
	fileTypes: fileTypes,
	firstLineMatch: firstLineMatch,
	foldingStartMarker: foldingStartMarker,
	foldingStopMarker: foldingStopMarker,
	patterns: patterns,
	repository: repository
};

export { $schema, apl_tmLanguage as default, fileTypes, firstLineMatch, foldingStartMarker, foldingStopMarker, name, patterns, repository, scopeName };
